/*
 * leds-tlc59116.c - RGB LED Driver
 *
 * crestron:
 * 	Rjk 	: Jul 12 2013	: file creation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 *
 */

#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include <linux/leds.h>
#include <linux/slab.h>
#include <linux/pm.h>
#include <linux/crestron.h>
#include "crest_tlc59116.h"
#include <linux/gpio.h>
#include <linux/fsl_devices.h>

#include <linux/proc_fs.h>  /* Necessary because we use proc fs */
#include <asm/uaccess.h>    /* for copy_*_user */

#define BACKLIGHT_FILENAME           "tlc59116"
#define PROCFS_MAX_SIZE     512
#define MAX_NUM_OF_LIGHT_GROUPS 2

enum led_colors {
	OFF,
	GREEN,  //use this for white as well when only one color
	WHITE = 1,
	RED,
	BLUE,
};

#define DEFAULT_COLOR OFF

#define TLC59116_MAX_LEDS			(16)	/* Maximum number of LEDs */
#define MIN_BRIGHTNESS				(0)
#define MAX_PROC_BRIGHTNESS			100
#define MAX_BRIGHTNESS				(0xFF)
#define DEFAULT_BRIGHTNESS 			(0)

struct color_state {
	u8 r;
	u8 g;
	u8 b;
	u8 w;
};

struct tlc59116_led {
	int			id;
//	u8			chan_nr;
//	u8			led_current;
//	u8			max_current;
	struct led_classdev	cdev;
	struct work_struct	brightness_work;
	u8			brightness;
};
struct tlc59116_chip {
	struct tlc59116_platform_data *pdata;
	struct mutex		lock; /* Serialize control */
	struct i2c_client	*client;
	struct tlc59116_led	leds[TLC59116_MAX_LEDS];

	int procBrightness;
	int brightness;
	int color;
	struct color_state colorState;

//	u8			num_channels;
//	u8			num_leds;
};
static struct tlc59116_chip* gChip[MAX_NUM_OF_LIGHT_GROUPS];
static struct proc_dir_entry *BackLight_File;

static inline struct tlc59116_led *cdev_to_led(struct led_classdev *cdev)
{
	return container_of(cdev, struct tlc59116_led, cdev);
}

static inline struct tlc59116_chip *led_to_tlc59116(struct tlc59116_led *led)
{
	return container_of(led, struct tlc59116_chip,
			    leds[led->id]);
}

static int tlc59116_write_reg(struct i2c_client *client, u8 reg, u8 val)
{
	int ret = i2c_smbus_write_byte_data(client, reg, val);

	if (ret < 0)
		dev_err(&client->dev, "%s: reg 0x%x, val 0x%x, err %d\n",
			__func__, reg, val, ret);
	return ret;
}

static int tlc59116_read_reg(struct i2c_client *client, int reg)
{
	int ret = i2c_smbus_read_byte_data(client, reg);

	if (ret < 0)
		dev_err(&client->dev, "%s: reg 0x%x, err %d\n",
			__func__, reg, ret);
	return ret;
}

// Format of payload of an I2C packet:  To write a whole page of NVRAM data.
struct i2cPacketWriteLeds
{
	u8 addr;
	u8 data[TLC59116_MAX_LEDS];
};

static int tlc59116_write_block(struct i2c_client *client, u8 reg, u8* val, int len)
{
	int retval = 0;
	struct i2c_msg msg;
	struct i2cPacketWriteLeds packet = {0};

	packet.addr = reg | KBL_TLC_AUTO_INC;
	memcpy(packet.data , val, len);

	msg.addr = client->addr;
	msg.flags = 0;
	msg.len = sizeof(packet);
	msg.buf = (u8*)&packet;

	retval = i2c_transfer(client->adapter, &msg, 1);
	if (retval < 0)
			dev_err(&client->dev, "%s: reg 0x%x, val 0x%x, err %d\n",
				__func__, reg, *val, retval);
	return retval;
}
#if 0 //currently unused
static int  GetIndividualBrightness(struct tlc59116_chip *chip, int led)
{
	int brightness = -1;
	if(led < TLC59116_MAX_LEDS)
		brightness = tlc59116_read_reg(chip->client, KBL_TLC_REG_PWM0 + led);
	return brightness;
}
#endif
static int  SetIndividualBrightness(struct tlc59116_chip *chip, int led, u8 brightness)
{
	int retval = 0;
	if(led < TLC59116_MAX_LEDS)
	{
		retval = tlc59116_write_reg(chip->client, KBL_TLC_REG_PWM0 + led, brightness);
	}
	return retval;
}

static int  SetColorBrightness(struct tlc59116_chip *chip)
{
	u8 brightness[TLC59116_MAX_LEDS];
	int i;
	for(i = 0; i < chip->pdata->leds; i+=chip->pdata->colors)
	{
		if(chip->pdata->colors == 1)
			brightness[i] = chip->colorState.w;
		else if(chip->pdata->colors == 3)
		{
			brightness[i] = chip->colorState.g;
			brightness[i+1] = chip->colorState.b;
			brightness[i+2] = chip->colorState.r;
		}
	}
	return tlc59116_write_block(chip->client, KBL_TLC_REG_PWM0, brightness, TLC59116_MAX_LEDS);
}

static int  SetGroupBrightness(struct tlc59116_chip *chip, u8 brightness)
{
	return tlc59116_write_reg(chip->client, KBL_TLC_REG_GRPPWM, brightness);
}

static ssize_t colors_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct tlc59116_chip *chip = i2c_get_clientdata(client);

	return sprintf(buf, "%u\n", chip->pdata->colors);
}

static DEVICE_ATTR(colors, S_IRUGO | S_IWUSR, colors_show, NULL);

static ssize_t numberOfLeds_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct tlc59116_chip *chip = i2c_get_clientdata(client);

	return sprintf(buf, "%u in %u group(s)\n", chip->pdata->leds, chip->pdata->led_groups);
}

static DEVICE_ATTR(numberOfLeds, S_IRUGO | S_IWUSR, numberOfLeds_show, NULL);

static ssize_t colorBrightness_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct tlc59116_chip *chip = i2c_get_clientdata(client);

	if(chip->pdata->colors == 1)
		return sprintf(buf, "%u\n", chip->colorState.w);
	else if(chip->pdata->colors == 3)
		return sprintf(buf, "r:%u g:%u b:%u\n", chip->colorState.r, chip->colorState.g, chip->colorState.b);
	else
		return sprintf(buf, "invalid number of colors\n");
}

static ssize_t colorBrightness_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct tlc59116_chip *chip = i2c_get_clientdata(client);
	if(chip->pdata->colors == 1)
	{
		sscanf(buf,"%hhu",&chip->colorState.w);
		chip->brightness = chip->colorState.w;
	}
	else if(chip->pdata->colors == 3)
	{
		sscanf(buf,"%hhu %hhu %hhu",&chip->colorState.r,&chip->colorState.g,&chip->colorState.b);
		chip->brightness = max((int)chip->colorState.r,(int)chip->colorState.g);
		chip->brightness = max(chip->brightness,(int)chip->colorState.b);
	}

	//to keep in sync with proc file
	if(chip->brightness == 0)
		chip->color = OFF;
	else if(chip->brightness == chip->colorState.r)
		chip->color = RED;
	else if(chip->brightness == chip->colorState.g)
		chip->color = GREEN;
	else if(chip->brightness == chip->colorState.b)
		chip->color = BLUE;
	else if(chip->brightness == chip->colorState.w)
		chip->color = WHITE;
	chip->procBrightness = chip->brightness * MAX_PROC_BRIGHTNESS / MAX_BRIGHTNESS;

	SetColorBrightness(chip);
	return count;
}

static DEVICE_ATTR(colorBrightness, S_IRUGO | S_IWUSR, colorBrightness_show, colorBrightness_store);

static ssize_t groupBrightness_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	u8 brightness = -1;
	struct i2c_client *client = to_i2c_client(dev);
	struct tlc59116_chip *chip = i2c_get_clientdata(client);

	brightness = tlc59116_read_reg(chip->client, KBL_TLC_REG_GRPPWM);
	return sprintf(buf, "%u\n", brightness);
}

static ssize_t groupBrightness_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
	int brightness;
	struct i2c_client *client = to_i2c_client(dev);
	struct tlc59116_chip *chip = i2c_get_clientdata(client);
	sscanf(buf,"%u",&brightness);
	SetGroupBrightness(chip, brightness);
	return count;
}

static DEVICE_ATTR(groupBrightness, S_IRUGO | S_IWUSR, groupBrightness_show, groupBrightness_store);

static struct attribute *tlc_attributes[] = {
	&dev_attr_colors.attr,
	&dev_attr_numberOfLeds.attr,
	&dev_attr_colorBrightness.attr,
    &dev_attr_groupBrightness.attr,
    NULL
};
static struct attribute_group tlc_attribute_group = {
    .attrs = tlc_attributes
};

static void tlc59116_brightness_set(struct led_classdev *led_cdev,
				     enum led_brightness brt_val)
{
	struct tlc59116_led *led = cdev_to_led(led_cdev);
	struct tlc59116_chip *chip = led_to_tlc59116(led);

	if(brt_val > MAX_BRIGHTNESS)
		brt_val = MAX_BRIGHTNESS;
	else if(brt_val < 0)
		brt_val = 0;

	SetIndividualBrightness(chip,led->id, brt_val);
}

/** * This funtion is called when the /proc brightness file is read */
static ssize_t tlc59116_read(struct file *file,  /* see include/linux/fs.h   */
                             char *buffer,      /* buffer to fill with data */
                             size_t length,     /* length of the buffer     */
                             loff_t * offset)
{
	static int finished = 0;
	char backLight_buffer[PROCFS_MAX_SIZE];
	char * p = backLight_buffer;
	unsigned long backLight_buffer_size = 0;
	struct tlc59116_chip* chip = gChip[0];

	int ledID;

	/* needed to stop from continuously printing */
	if ( finished == 1 ) { finished=0; return 0; }
	finished = 1;

	p += sprintf ( p, "NUM_OF_LIGHTS=%04X\n" , chip->pdata->led_groups );
	p += sprintf ( p, "NUM_OF_COLORS=%04d\n", chip->pdata->colors);
	p += sprintf ( p, "DEFAULT_COLOR=%04d\n", DEFAULT_COLOR );
	p += sprintf ( p, "MIN_BRIGHTNESS=%04d\n" ,MIN_BRIGHTNESS);
	p += sprintf ( p, "MAX_BRIGHTNESS=%04d\n" ,MAX_PROC_BRIGHTNESS);
	p += sprintf ( p, "DEFAULT_BRIGHTNESS=%04d\n", DEFAULT_BRIGHTNESS);
	for(ledID  = 0; ledID < chip->pdata->led_groups; ledID++)
	{
		chip = gChip[ledID];
		if(chip != NULL)
		{
			p += sprintf ( p, "BRIGHTNESS%02d= %04d\n", ledID, chip->procBrightness );
			p += sprintf ( p, "COLOR%02d= %04d\n", ledID, chip->color );
		}
	}
	backLight_buffer_size = p-backLight_buffer;

	/*
	 * We use put_to_user to copy the string from the kernel's
	 * memory segment to the memory segment of the process
	 * that called us. get_from_user, BTW, is
	 * used for the reverse.
	 */
	if ( copy_to_user(buffer, backLight_buffer, backLight_buffer_size) ) {
			return -EFAULT;
	}

	return backLight_buffer_size;
}


static ssize_t tlc59116_write(struct file *file, const char *buffer, size_t len, loff_t * off)
{
  	char* tag = NULL;
  	char* value = NULL;
  	char backLight_buffer[PROCFS_MAX_SIZE];
  	char* tempPtr = backLight_buffer;
  	unsigned long backLight_buffer_size = 0;
  	struct tlc59116_chip* chip = gChip[0];
	int ledID;
	int val;
	/* get buffer size */
	backLight_buffer_size = len;
	if (backLight_buffer_size > PROCFS_MAX_SIZE ) {
		backLight_buffer_size = PROCFS_MAX_SIZE;
	}

	/* write data to the buffer */
	if ( copy_from_user(backLight_buffer, buffer, backLight_buffer_size) )
	{
		return -EFAULT;
	}
	tag = strsep ( &tempPtr, "=" );

	if ( strncmp ( tag, "BRIGHTNESS", 10 ) == 0 )
	{
		value = strsep ( &tempPtr, "=" );
		sscanf ( tag, "BRIGHTNESS%d", &ledID);
		sscanf ( value, "%d", &val);

		if(ledID < chip->pdata->led_groups)
		{
			chip = gChip[ledID];
		}
		if(chip != NULL)
		{
			chip->procBrightness = val;
			if(chip->procBrightness > MAX_PROC_BRIGHTNESS)
				chip->procBrightness = 100;

			chip->brightness = chip->procBrightness * MAX_BRIGHTNESS / MAX_PROC_BRIGHTNESS;
			if(chip->pdata->colors == 1)
				chip->colorState.w = (chip->color == WHITE)?chip->brightness:0;
			else if(chip->pdata->colors == 3)
			{
				chip->colorState.r = (chip->color == RED)?chip->brightness:0;
				chip->colorState.g = (chip->color == GREEN)?chip->brightness:0;
				chip->colorState.b = (chip->color == BLUE)?chip->brightness:0;
			}
			SetColorBrightness(chip);
		}

	}
	if ( strncmp ( tag, "COLOR", 5 ) == 0 )
	{
		value = strsep ( &tempPtr, "=" );
		sscanf ( value, "%d", &val );
		sscanf ( tag, "COLOR%d", &ledID);
		if(ledID < chip->pdata->led_groups)
			chip = gChip[ledID];
		if(chip != NULL)
		{
			chip->color = val;
			if(chip->color > chip->pdata->colors)
				chip->color = OFF;

			if(chip->pdata->colors == 1)
				chip->colorState.w = (chip->color == WHITE)?chip->brightness:0;
			else if(chip->pdata->colors == 3)
			{
				chip->colorState.r = (chip->color == RED)?chip->brightness:0;
				chip->colorState.g = (chip->color == GREEN)?chip->brightness:0;
				chip->colorState.b = (chip->color == BLUE)?chip->brightness:0;
			}
			SetColorBrightness(chip);
		}
	}

	return backLight_buffer_size;
}

static int module_permission(struct inode *inode, int op, unsigned int foo)
{
	return 0;
}

static struct inode_operations Inode_Ops_File = {
	.permission = module_permission,	/* check for permissions */
};

/*
 * The file is opened - we don't really care about
 * that, but it does mean we need to increment the
 * module's reference count.
 */
int tlc59116_open_procfs(struct inode *inode, struct file *file)
{
	try_module_get(THIS_MODULE);
	return 0;
}

/*
 * The file is closed - again, interesting only because
 * of the reference count.
 */
int tlc59116_close_procfs(struct inode *inode, struct file *file)
{
	module_put(THIS_MODULE);
	return 0;               /* success */
}

static struct file_operations File_Ops_BackLight_File = {
	.read    = tlc59116_read,
	.write   = tlc59116_write,
	.open    = tlc59116_open_procfs,
	.release = tlc59116_close_procfs,
};

int KBL_Reg_init(struct i2c_client *client,
		const struct i2c_device_id *id)
{
	struct tlc59116_chip *chip = i2c_get_clientdata(client);
	int led_enable_mask = 0; //each led gets 2 bits
	int i;
	int ret=-1;

	tlc59116_write_reg(client, KBL_TLC_REG_MODE1, 0x00);// Set normal mode, etc.
	tlc59116_write_reg(client, KBL_TLC_REG_MODE2, 0x80);// Clear error status flags.

	/* Detect tlc59116: The initial Keys FIFO value is '0x3F' */
	ret = tlc59116_read_reg(client, KBL_TLC_REG_MODE1);
	if (ret < 0) {
		dev_err(&client->dev, "failed to detect tlc59116 device\n");
		return -ENODEV;
	}
	ret = tlc59116_read_reg(client, KBL_TLC_REG_MODE2);
	if (ret < 0) {
			dev_err(&client->dev, "failed to detect tlc59116 device\n");
			return -ENODEV;
	}

	for(i = 0; i < chip->pdata->leds; i++)
	{
		led_enable_mask = led_enable_mask << 2;
		led_enable_mask |= KBL_LEDOUT_GROUP;
	}

	tlc59116_write_reg(client, KBL_TLC_REG_MODE1, 0x00);// Set normal mode, etc.
	tlc59116_write_reg(client, KBL_TLC_REG_MODE2, 0x80);// Clear error status flags.
	tlc59116_write_reg(client, KBL_TLC_REG_MODE2, 0x00);// Enable error status flags.
	tlc59116_write_reg(client, KBL_TLC_REG_GRPPWM, 0);// Group brightness.
	tlc59116_write_reg(client, KBL_TLC_REG_LEDOUT0, (led_enable_mask >> 0) & 0xFF);// LED 0..3 state.
	tlc59116_write_reg(client, KBL_TLC_REG_LEDOUT1, (led_enable_mask >> 8) & 0xFF);// LED 4..7 state.
	tlc59116_write_reg(client, KBL_TLC_REG_LEDOUT2, (led_enable_mask >> 16) & 0xFF); // LED 8..11 state.
	tlc59116_write_reg(client, KBL_TLC_REG_LEDOUT3, (led_enable_mask >> 24) & 0xFF); // LED 12..15 state.
	tlc59116_write_reg(client, KBL_TLC_REG_PWM0, 0 /*255*/); // LED 0 brightness.
	tlc59116_write_reg(client, KBL_TLC_REG_PWM1, 0 /*255*/); // LED 1 brightness.
	tlc59116_write_reg(client, KBL_TLC_REG_PWM2, 0 /*255*/); // LED 2 brightness.
	tlc59116_write_reg(client, KBL_TLC_REG_PWM3, 0 /*255*/); // LED 3 brightness.
	tlc59116_write_reg(client, KBL_TLC_REG_PWM4, 0 /*255*/); // LED 4 brightness.
	tlc59116_write_reg(client, KBL_TLC_REG_PWM5, 0 /*255*/); // LED 5 brightness.
	tlc59116_write_reg(client, KBL_TLC_REG_PWM6, 0 /*255*/); // LED 6 brightness.
	tlc59116_write_reg(client, KBL_TLC_REG_PWM7, 0 /*255*/); // LED 7 brightness.
	tlc59116_write_reg(client, KBL_TLC_REG_PWM8, 0 /*255*/); // LED 8 brightness.
	tlc59116_write_reg(client, KBL_TLC_REG_PWM9, 0 /*255*/); // LED 9 brightness.
	tlc59116_write_reg(client, KBL_TLC_REG_PWM10, 0 /*255*/); // LED 10 brightness.
	tlc59116_write_reg(client, KBL_TLC_REG_PWM11, 0 /*255*/); // LED 11 brightness.
	tlc59116_write_reg(client, KBL_TLC_REG_GRPPWM, 255); // Group PWM brightness.
	return 0;
}

static int __devinit tlc59116_probe(struct i2c_client *client,
			const struct i2c_device_id *id)
{
	struct tlc59116_chip *chip;
	struct tlc59116_platform_data *pdata;
	int ret=-ENODEV, i = 0;
	char name[32];

	chip = kzalloc(sizeof(struct tlc59116_chip), GFP_KERNEL);
	if (!chip) {
		dev_err(&client->dev, "failed to allocate driver data\n");
		goto err_out;
	}

	chip->client = client;
	chip->brightness = DEFAULT_BRIGHTNESS;
	chip->color = DEFAULT_COLOR;

	pdata = chip->pdata = client->dev.platform_data;
	i2c_set_clientdata(client, chip);

	if(chip->pdata->led_groups > MAX_NUM_OF_LIGHT_GROUPS)
	{
		printk(KERN_ERR "%s() - ERROR: unsupported number of led groups\n", __FUNCTION__);
		goto err_i2c;
	}

	if(chip->pdata->leds > TLC59116_MAX_LEDS)
	{
		printk(KERN_ERR "%s() - ERROR: unsupported number of leds\n", __FUNCTION__);
		goto err_i2c;
	}

	if(pdata->id > chip->pdata->led_groups)
		goto err_i2c;

	ret = KBL_Reg_init(client, id);
	if(ret)
		goto err_i2c;

	for(i = 0; i < TLC59116_MAX_LEDS; i++)
	{
		snprintf(name, sizeof(name), "led%d-%d", pdata->id, i);

		chip->leds[i].id = i;
		chip->leds[i].cdev.name = name;
		chip->leds[i].cdev.brightness_set = tlc59116_brightness_set;
		ret = led_classdev_register(&client->dev, &chip->leds[i].cdev);
		if (ret) {
			dev_err(&client->dev, "failed to register LED %d\n",i);
			goto err;
		}
	}

	ret = sysfs_create_group(&client->dev.kobj, &tlc_attribute_group);
	if (ret < 0)
	{
		printk(KERN_ERR "%s() - ERROR: sysfs_create_group() failed: %d\n", __FUNCTION__, ret);
		goto err_register_dev;
	}
	/* create the /proc file for Brightness bitmask */
	BackLight_File = create_proc_entry(BACKLIGHT_FILENAME, 0644, NULL);
	if (BackLight_File == NULL){
		printk(KERN_ALERT "Error: Could not initialize /proc/%s\n", BACKLIGHT_FILENAME);
		goto err_create_proc;
	}
	else
	{
		//BackLight_File->owner = THIS_MODULE;
		BackLight_File->proc_iops = &Inode_Ops_File;
		BackLight_File->proc_fops = &File_Ops_BackLight_File;
		BackLight_File->mode = S_IFREG | S_IRUGO | S_IWUSR;
		BackLight_File->uid = 0;
		BackLight_File->gid = 0;
		BackLight_File->size = 80;
	}
	gChip[pdata->id] = chip;
	return 0;
/*
failed_unregister_dev_file:

failed_free:

*/
err_create_proc:
	sysfs_remove_group(&client->dev.kobj, &tlc_attribute_group);
err_register_dev:
	i = 12;
err:
	for(i--; i >= 0; i--)
	{
		led_classdev_unregister(&chip->leds[i].cdev);
	}
err_i2c:
	i2c_set_clientdata(client, NULL);
	kfree(chip);
err_out:
	return ret;
}
#if 0
static int __exit tlc59116_remove(struct i2c_client *client)
{
	struct tlc59116_led *led = i2c_get_clientdata(client);
	int i;

	gpio_set_value(led->pdata->reset_gpio, 0);
	tlc59116_unregister_led_classdev(led);
	if (led->adf_on)
		tlc59116_disable_adv_conf(led);
	for (i = 0; i < ARRAY_SIZE(tlc59116_attributes); i++)
		device_remove_file(&led->client->dev, bd2802_attributes[i]);
	kfree(led);

	return 0;
}

#ifdef CONFIG_PM

static void tlc59116_restore_state(struct tlc59116_led *led)
{
	int i;

	for (i = 0; i < LED_LOCATIONS; i++) {
		if (led->led[i].r)
			tlc59116_turn_on(led, i, RED, led->led[i].r);
		if (led->led[i].g)
			tlc59116_turn_on(led, i, GREEN, led->led[i].g);
		if (led->led[i].b)
			tlc59116_turn_on(led, i, BLUE, led->led[i].b);
	}
}

static int tlc59116_suspend(struct device *dev)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct tlc59116_led *led = i2c_get_clientdata(client);

	gpio_set_value(led->pdata->reset_gpio, 0);

	return 0;
}

static int tlc59116_resume(struct device *dev)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct tlc59116_led *led = i2c_get_clientdata(client);

	if (!tlc59116_is_all_off(led) || led->adf_on) {
		tlc59116_reset_cancel(led);
		tlc59116_restore_state(led);
	}

	return 0;
}


#endif

#endif

static const struct i2c_device_id tlc59116_id[] = {
	{ "tlc59116", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, tlc59116_id);

static struct i2c_driver tlc59116_i2c_driver = {
	.driver	= {
		.name	= "tlc59116",
		//.pm	= TLC59116_PM,
	},
	.probe		= tlc59116_probe,
	.remove		= __exit_p(tlc59116_remove),
	.id_table	= tlc59116_id,
};

static int __init tlc59116_init(void)
{
	return i2c_add_driver(&tlc59116_i2c_driver);
}
module_init(tlc59116_init);

static void __exit tlc59116_exit(void)
{
	i2c_del_driver(&tlc59116_i2c_driver);
}
module_exit(tlc59116_exit);

MODULE_AUTHOR("Crestron Electronics");
MODULE_DESCRIPTION("tlv59116 LED driver");
MODULE_LICENSE("GPL v2");
